################################################################################
# implements the OS list for the files dialog
################################################################################


# NOTES - 
#    - Need to Update SVD-Control.tcl to initialize OSListDir (remove
#        the references to OSListFile/Alt which was used before).
#    - Need to update SVD-Control.tcl to specify the OSListExtension
#        (this extension does NOT have the preceeding period)

########################################################################
# loadOSList() -	Loads the OS list from all of the .osl files in
#                       the directories given in the .ini file.
#
# NOTE - OSListFile("icon") lists the gif file for the floppy icon
#      - OSListFile("exts") lists the file extensions for this floppy type
#      - OSListFile("name") is the pretty display name of this floppy type
########################################################################
proc loadOSList {floppytype} {
    global  OSListSimple
    global  OSListOrder
    global  OSListType
    global  OSListFS
    global  OSListName
    global  OSListFile

    # clear the last list

    unset -nocomplain OSListOrder
    unset -nocomplain OSListType
    unset -nocomplain OSListFS
    unset -nocomplain OSListName
    unset -nocomplain OSListFile

    set OSListDir [iniGetAttribute "Program" "OSListDir"]
    set OSListExtension [iniGetAttribute "Program" "OSListExtension"]

    foreach {i} [glob -directory $OSListDir -nocomplain "*.$OSListExtension"] {
	readOSList $floppytype $i
    }

    # note that OSListFile("icon") lists the gif file for the floppy icon
    # note that OSListFile("exts") lists the file extensions for this floppy type
    # note that OSListFile("name") is the pretty display name of this floppy type

    # now create the simple list that is used to display user choices of OS file

    set OSListSimple ""
    for { set i 1 } { $i <= [array size OSListOrder] } {incr i} {

	# those entries with "(opt)" as the file system refer to internal
	# options that relate to the given floppy type and shouldn't be
	# listed for the user to select

	if { $OSListFS($OSListOrder($i)) != "(opt)" } {
	    lappend OSListSimple $OSListOrder($i)
	}
    }
}

########################################################################
# readOSList() -	Reads in the OSList for display and generation.
#                       The list is appended to, and includes only those
#                       entries that match the given type of floppy.
########################################################################
proc readOSList {floppytype filename} {
    global  OSListOrder
    global  OSListName
    global  OSListType
    global  OSListFS
    global  OSListFile
    global  tcl_platform

    set fstype ""
    set name ""
    set file ""
    set type ""

    if { ! [file readable $filename] } {
	return
    }

    set osfile [open $filename "r"]

    set i [array size OSListOrder]
    incr i

    while { [gets $osfile line] >= 0 } {

	# first strip the leading spaces from a line
	regexp -line {^[ \t]*(.*)} $line ignoreMatch line

	# now punt comment lines
	regexp -line {^#.*} $line ignoreMatch line

	# ignore blank lines (which now includes comment lines)
	if { $line != "" } {
	    regexp -line {^([^	]*) *[	]+ *([^	]*) *[	]+ *([^	]*) *[	]+ *([^	]*)[ 	]*$} \
		    $line ignoreMatch type fstype name file

	    # note that this OSList is organized by the display "name"
	    # 'cause that's how it is used anyway - this name must be
	    # unique within a given floppy type

	    if { $type == $floppytype } {
		set OSListOrder($i) $name
		set OSListName($name) $name
		set OSListType($name) $type
		set OSListFS($name) $fstype
		set OSListFile($name) $file

		# note that this will also flip "/" characters in the description
		# of the format...but that really shouldn't matter much

		if { $tcl_platform(platform) == "windows" } {
		    set OSListFile($name) [join [split $OSListFile($name) "/"] "\\"]
		}
		incr i
	    }
	}
    }

    close $osfile
}

########################################################################
# floppyExtsUC() -	Extends the given list (assumed to be a list of
#                       extensions) by adding uppercase versions.
########################################################################
proc floppyExtsUC {extlist} {
    set returnlist ""

    foreach {e} $extlist {
	lappend returnlist $e
	lappend returnlist [string toupper $e]
    }

    return $returnlist
}


########################################################################
# floppyExts() -	Looks as all of the .osl files and return a list
#                       of all exentions.
########################################################################
proc floppyExts {} {
    set returnlist ""
    set OSListDir [iniGetAttribute "Program" "OSListDir"]
    set OSListExtension [iniGetAttribute "Program" "OSListExtension"]

    foreach {i} [glob -directory $OSListDir -nocomplain "*.$OSListExtension"] {
	foreach e [readExts $i] {
	    if { [lsearch -exact $returnlist $e] == -1 } {
		lappend returnlist $e
	    }
	}
    }

    return $returnlist
}


########################################################################
# readExts() -	given a file, read all of the extensions in that file.
#               Simply return a list of those extensions.  Duplicates are
#               filtered out.
########################################################################
proc readExts {filename} {
    set returnlist ""
    set fstype ""
    set name ""
    set file ""
    set type ""

    if { ! [file readable $filename] } {
	return
    }

    set osfile [open $filename "r"]

    while { [gets $osfile line] >= 0 } {

	# first strip the leading spaces from a line
	regexp -line {^[ \t]*(.*)} $line ignoreMatch line

	# now punt comment lines
	regexp -line {^#.*} $line ignoreMatch line

	# ignore blank lines (which now includes comment lines)
	if { $line != "" } {
	    regexp -line {^([^	]*) *[	]+ *([^	]*) *[	]+ *([^	]*) *[	]+ *([^	]*)[ 	]*$} \
		    $line ignoreMatch type fstype name file

	    if { $fstype == "(opt)" && $name == "exts"} {
		foreach e $file {
		    if { [lsearch -exact $returnlist $e] == -1 } {
			lappend returnlist $e
		    }
		}
	    }
	}
    }

    close $osfile

    return $returnlist
}

########################################################################
# floppyTypeOSList() -	generates a list of floppy types by looking at
#                       at all of the OSList files.
########################################################################
proc floppyTypeOSList {} {
    global FloppyDescriptions

    unset -nocomplain FloppyTypes
    unset -nocomplain FloppyDescriptions

    set OSListDir [iniGetAttribute "Program" "OSListDir"]
    set OSListExtension [iniGetAttribute "Program" "OSListExtension"]

    foreach {i} [glob -directory $OSListDir -nocomplain "*.$OSListExtension"] {
	readTypes $i
    }
}


########################################################################
# readTypes() - given a file, read in all of the types in that file.
#               One global variable is set.
#
#      FloppyDescriptions() - array of descriptions indexed by FloppyTypes
#
#               If run multiple times, the array is appended.
#
#         Knows that the descriptive name entry in the list is set with
#         the fstype of "(opt)" and name of "name".
########################################################################
proc readTypes {filename} {
    global  FloppyDescriptions

    set fstype ""
    set name ""
    set file ""
    set type ""

    if { ! [file readable $filename] } {
	return
    }

    set osfile [open $filename "r"]

    while { [gets $osfile line] >= 0 } {

	# first strip the leading spaces from a line
	regexp -line {^[ \t]*(.*)} $line ignoreMatch line

	# now punt comment lines
	regexp -line {^#.*} $line ignoreMatch line

	# ignore blank lines (which now includes comment lines)
	if { $line != "" } {
	    regexp -line {^([^	]*) *[	]+ *([^	]*) *[	]+ *([^	]*) *[	]+ *([^	]*)[ 	]*$} \
		    $line ignoreMatch type fstype name file

	    if { $fstype == "(opt)" && $name == "name" } {
		set FloppyDescriptions($type) $file
	    }
	}
    }

    close $osfile
}

